#!/usr/bin/perl -w
##
## (c)  Copyright  IBM Corp.  2004  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
## Module History:
##  -00 10/29/2004 John Thornton   D61C initial release
##
##

#
# Includes
#
use Getopt::Long;

##############################################################################
#####                          GLOBAL VARIABLES                          #####
##############################################################################

# Trace Constants
$TRC_T = "XMCFSPST:";
$TRC_D = "XMCFSPSD:";
$TRC_F = "XMCFSPSF:";

# Error Constants
$ERROR_NO_ERROR               =  0;
$ERROR_NOT_ROOT               =  1;
$ERROR_CANT_EXECUTE_SWAP      =  2;
$ERROR_UTILS_NOT_FOUND        =  3;
$ERROR_PERMS_FILE_ACCESS      =  4;
$ERROR_PERMS_CONTENT_ERROR    =  5;
$ERROR_PERMS_NO_ACTION        =  6;
$ERROR_PERMS_OWNER_FAILURE    =  7;
$ERROR_PERMS_GROUP_FAILURE    =  8;
$ERROR_PERMS_FLAGS_FAILURE    =  9;
$ERROR_HELP                   = 99;

# Boolean Constants
$TRUE  = 1;
$FALSE = 0;

# Program Data
$me = $0;
$me =~ s/^(.*)\///;
$me =~ s/.pl$//;
$traceCmd = "actzTrace";

##############################################################################
#####                          SHARED METHODS                            #####
##############################################################################

#
#
#
sub queryFileLocation
{
   my $this = $me . ".queryFileLocation()";
   my $file = shift;
   trace("$TRC_T -> $this - $file");
   my $consoleDir;
   # Many possibilities in the non-production world
   if ($ENV{'CONSOLE_PATH'})
   {
      $consoleDir = $ENV{'CONSOLE_PATH'}.'/';
      trace("$TRC_F    $this - used CONSOLE_PATH");
   }
   elsif ($ENV{'HWMCAHOME'})
   {
      $consoleDir = $ENV{'HWMCAHOME'}.'/';
      trace("$TRC_F    $this - used HWMCAHOME");
   }
   else
   {
      $consoleDir = "/console/";
      trace("$TRC_F    $this - used default");
   }
   # Change any "//" to "/"
   $consoleDir =~ s/\/+/\//g;
   trace("$TRC_F    $this - console root is $consoleDir");
   if ( -x "$consoleDir/hmcfunctions" )
   {
     $path = `. hmcfunctions; queryFileLocation $file`;
     # Remove eol
     chomp($path);
   }
   else
   {
     # Make our best guess
     $path = $consoleDir . "data/";
     trace("$TRC_F    $this - hmcfunctions not found - only guessing here");
     my $hex_rc = sprintf("0x%08X", $ERROR_UTILS_NOT_FOUND);
     `logerror 0xAA 0x02 0x0000 0xE562 $hex_rc`;
   }
   trace("$TRC_T <- $this - $path");
   return($path);
}

#
#
#
sub setHozd
{
   my $this = $me . ".setHosed()";
   my $msg = shift;
   trace("$TRC_T -> $this - message is $msg");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmhozd.dat";

   my $path = queryFileLocation($filename);
   my $hozdfile = $path . $filename;
   # Change any "//" to "/"
   $hozdfile =~ s/\/+/\//g;
   trace("$TRC_F    $this - hosed file is $hozdfile");

   open(HOZDFILE, ">> " . $hozdfile);
   print HOZDFILE ($msg . "\n");
   close(HOZDFILE);

   trace("$TRC_F    $this - issue the HOZD refcode");
   `logerror 0xAA 0x02 0x0000 0xE563 0x0000DEAD`;

   trace("$TRC_T <- $this");
   return;
}

#
#
#
sub trace
{
    my $text = shift;
    chomp($text);

    if ("actzTrace" eq "$traceCmd")
    {
       system(("$traceCmd", "$text"));
    }
    else
    {
       print(STDOUT "$text\n");
    }
}

#
# usage
#
sub usage
{
    print(<<EOM);
Usage: iqzmShutdownProcessing [-help] [-tracecommand=cmd]
       this command is intended to be run from the manager script as the
       manager is shutting down
EOM
}

##############################################################################
#####                           FILE SWAPPING                            #####
##############################################################################

#
#
#
sub doFileSwapping
{
   my $this = $me . ".doFileSwapping()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmlock.sh";

   my $path = queryFileLocation($filename);
   my $swapfile = $path . $filename;
   # Change any "//" to "/"
   $swapfile =~ s/\/+/\//g;
   trace("$TRC_F    $this - swap file script is $swapfile");

   # Does the file exist?
   if ( -f "$swapfile" )
   {
      # Ensure the file is executable
      if ( !( -x "$swapfile" ))
      {
         # Not yet - set the permission bits
         my $count = chmod(0755, ($swapfile));
         if ($count < 1)
         {
            # chmod() failed, so we are toast for running the file.
            # This is fatal.
            trace("$TRC_F    $this - Unable to execute $swapfile");
            $rc = $ERROR_CANT_EXECUTE_SWAP;
            my $hex_rc = sprintf("0x%08X", $rc);
            `logerror 0xAA 0x02 0x0000 0xE561 $hex_rc`;
            setHozd("$this - Unable to execute $swapfile");
         }
      }

      if ($rc == $ERROR_NO_ERROR)
      {
         # Invoke the swap script
         trace("$TRC_F    $this - Running $swapfile");
         print(`$swapfile`);
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No file swapping data was found");
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

##############################################################################
#####                          LINK PROCESSING                           #####
##############################################################################

#
#
#
sub doLinkProcessing
{
   my $this = $me . ".doLinkProcessing()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

##############################################################################
#####                        PERMISSION CHANGING                         #####
##############################################################################

#
#
#
sub doPermissionChanging
{
   my $this = $me . ".doPermissionChanging()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmPermissions.dat";

   my $path = queryFileLocation($filename);
   my $permfile = $path . $filename;
   # Change any "//" to "/"
   $permfile =~ s/\/+/\//g;
   trace("$TRC_F    $this - permissions data file is $permfile");

   # Does the file exist?
   if ( -f "$permfile" )
   {
      unless (open(PERMFILE, $permfile))
      {
         trace("$TRC_F    $this - error opening permissions data file");
         $rc = $ERROR_PERMS_FILE_ACCESS;
      }

      if ($rc == $ERROR_NO_ERROR)
      {
         trace("$TRC_D    $this - reading permissions data file");
         my @permdata = <PERMFILE>;

         trace("$TRC_D    $this - closing permissions data file");
         close (PERMFILE);

         $rc = processPermissions(@permdata);
      }

      # Copy file to .old version
      my $target = $permfile . ".old";
      `cp -f $permfile $target`;

      # Copy file to .err version if error
      if ($rc != $ERROR_NO_ERROR)
      {
         $target = $permfile . ".err";
         `cp -f $permfile $target`;
      }

      # Erase the data file
      `rm $permfile`;

      if ($rc != $ERROR_NO_ERROR)
      {
         # Our processing failed, so we may not be able to access the files
         # where permissions were to be changed.
         # This is fatal.
         trace("$TRC_F    $this - Failure updating file permissions");
         my $hex_rc = sprintf("0x%08X", $rc);
         `logerror 0xAA 0x02 0x0000 0xE564 $hex_rc`;
         setHozd("$this - Unable to update file permissions");
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No permissions data file was found");
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processPermissions
{
   my $this = $me . ".processPermissions()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my @lines = @_;

   my $file    = "";
   my $owner   = "";
   my $group   = "";
   my $flags   = "";
   my $key     = "";
   my $value   = "";

   my $contentError = $FALSE;

   foreach $line (@lines)
   {
      # Remove the eol
      chomp($line);
      trace("$TRC_D    $this - line is [$line]");

      # Break the line into a key and its value
      ($key, $value) = $line =~ /^<(.*?)>(.*)/;
      # These elements can end up undefined, so handle that to
      # avoid runtime error messages
      unless (defined($key))   { $key = "";   }
      unless (defined($value)) { $value = ""; }

      trace("$TRC_D    $this - key is [$key], value is [$value]");
      if ($key eq "file")
      {
         trace("$TRC_D    $this - file is [$value]");
         my $newfile = $value;

         # process the data for the previous file
         if ($file ne "")
         {
            my $trc = processPermissionsForOneFile($file, $owner, $group, $flags);
            # Save the rc if it represents an error
            if ($trc != $ERROR_NO_ERROR)
            {
               $rc = $trc
            }
         }
         $owner   = "";
         $group   = "";
         $flags   = "";
         $file = $newfile;

      }
      elsif ($key eq "owner")
      {
         trace("$TRC_D    $this - owner is [$value]");
         my $newowner = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - owner value ignored [$newowner]");
            $contentError = $TRUE;
         }
         else
         {
            if ($owner ne "")
            {
               trace("$TRC_F    $this - multiple owner records for $file - using last value [$newowner]");
               $contentError = $TRUE;
            }
            $owner = $newowner;
         }
      }
      elsif ($key eq "group")
      {
         trace("$TRC_D    $this - group is [$value]");
         my $newgroup = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - group value ignored [$newgroup]");
            $contentError = $TRUE;
         }
         else
         {
            if ($group ne "")
            {
               trace("$TRC_F    $this - multiple group records for $file - using last value [$newgroup]");
               $contentError = $TRUE;
            }
            $group = $newgroup;
         }
      }
      elsif ($key eq "flags")
      {
         trace("$TRC_D    $this - flags are [$value]");
         my $newflags = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - flags value ignored [$newflags]");
            $contentError = $TRUE;
         }
         else
         {
            if ($flags ne "")
            {
               trace("$TRC_F    $this - multiple flags records for $file - using last value [$newflags]");
               $contentError = $TRUE;
            }
            $flags = $newflags;
         }
      }
      else
      {
         trace("$TRC_F    $this - unrecognized line [$line]");
         $contentError = $TRUE;
      }
   }

   # process the data for the previous (last) file
   if ($file ne "")
   {
      my $trc = processPermissionsForOneFile($file, $owner, $group, $flags);
      # Save the rc if it represents an error
      if ($trc != $ERROR_NO_ERROR)
      {
         $rc = $trc
      }
   }

   # If there has been no other error so far, make a note of any content error
   if ($rc == $ERROR_NO_ERROR)
   {
      if ($contentError == $TRUE)
      {
         $rc = $ERROR_PERMS_CONTENT_ERROR;
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processPermissionsForOneFile
{
   my $this = $me . ".processPermissionsForOneFile()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($file, $owner, $group, $flags) = @_;

   trace("$TRC_F    $this - parms are [$file], [$owner], [$group], [$flags]");

   my $actionRequired = $FALSE;

   # Change the owner
   if ($owner ne "")
   {
      $actionRequired = $TRUE;
      my $cmd = "chown $owner $file 2>&1";
      my $result = `$cmd`;
      my $trc = $? >> 8;
      if ($trc != 0)
      {
         # Had a problem. Shouldn't happen. This is bad.
         trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
         $rc = $ERROR_PERMS_OWNER_FAILURE;
         my $hex_rc = sprintf("0x%08X", $rc);
         `logerror 0xAA 0x02 0x0000 0xE565 $hex_rc`;
      }
      else
      {
         trace("$TRC_D    $this - command is [$cmd], result is [$result]");
      }
   }

   # Change the group
   if ($rc == $ERROR_NO_ERROR)
   {
      if ($group ne "")
      {
         $actionRequired = $TRUE;
         my $cmd = "chown :$group $file 2>&1";
         my $result = `$cmd`;
         my $trc = $? >> 8;
         if ($trc != 0)
         {
            # Had a problem. Shouldn't happen. This is bad.
            trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
            $rc = $ERROR_PERMS_GROUP_FAILURE;
            my $hex_rc = sprintf("0x%08X", $rc);
            `logerror 0xAA 0x02 0x0000 0xE565 $hex_rc`;
         }
         else
         {
            trace("$TRC_D    $this - command is [$cmd], result is [$result]");
         }
      }
   }

   # Change the permission bits
   if ($rc == $ERROR_NO_ERROR)
   {
      if ($flags ne "")
      {
         $actionRequired = $TRUE;
         my $cmd = "chmod $flags $file 2>&1";
         my $result = `$cmd`;
         my $trc = $? >> 8;
         if ($trc != 0)
         {
            # Had a problem. Shouldn't happen. This is bad.
            trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
            $rc = $ERROR_PERMS_FLAGS_FAILURE;
            my $hex_rc = sprintf("0x%08X", $rc);
            `logerror 0xAA 0x02 0x0000 0xE565 $hex_rc`;
         }
         else
         {
            trace("$TRC_D    $this - command is [$cmd], result is [$result]");
         }
      }
   }

   # Was there something to do?
   if ($actionRequired == $FALSE)
   {
      trace("$TRC_F    $this - no action was specified for file $file");
      my $hex_rc = sprintf("0x%08X", $ERROR_PERMS_NO_ACTION);
      `logerror 0xAA 0x02 0x0000 0xE565 $hex_rc`;
      # This is not a fatal error.
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

##############################################################################
#####                          MAINLINE CODE                             #####
##############################################################################

# Process the arguments
GetOptions(
      'help'            =>\$help,
      'tracecommand:s'  =>\$traceCmd,
);

trace("$TRC_T -> $me(@ARGV)");

my $rc = $ERROR_NO_ERROR;

unless($> == 0)
{
   trace("$TRC_F    $me() - this must be run with root authority");
   my $hex_rc = sprintf("0x%08X", $ERROR_NOT_ROOT);
   `logerror 0xAA 0x02 0x0000 0xE560 $hex_rc`;
}

if ($help)
{
   usage();
   $rc = $ERROR_HELP;
}

# Run each of the functions, unless an error interferes.

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doFileSwapping();
}

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doLinkProcessing();
}

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doPermissionChanging();
}

trace("$TRC_T <- $me() - rc = $rc");

exit($rc);

